<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';
require_once '../helpers/permission_modal.php';

// Check if user has any role management permissions
$rolePermissions = [
    'manage_roles', 'view_roles', 'add_role', 'edit_role', 'delete_role'
];

if (!hasAnyPermission($rolePermissions)) {
    $_SESSION['error'] = "You don't have permission to access role management.";
    header('Location: dashboard.php');
    exit;
}

$activePage = 'roles';

$success = '';
$error = '';

// Handle role creation/update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    try {
        if ($_POST['action'] === 'create') {
            $name = trim($_POST['name']);
            $description = trim($_POST['description']);
            $permissions = isset($_POST['permissions']) ? $_POST['permissions'] : [];

            // Insert new role
            $stmt = $pdo->prepare("INSERT INTO roles (name, description) VALUES (?, ?)");
            $stmt->execute([$name, $description]);
            $role_id = $pdo->lastInsertId();

            // Insert role permissions
            if (!empty($permissions)) {
                $stmt = $pdo->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
                foreach ($permissions as $permission_id) {
                    $stmt->execute([$role_id, $permission_id]);
                }
            }

            logSystemAction('role_created', 'Created role: ' . $name);
            $success = 'Role created successfully!';
        } elseif ($_POST['action'] === 'update') {
            // Check if trying to modify super admin role
            $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = ?");
            $stmt->execute([$_POST['role_id']]);
            $role = $stmt->fetch();
            
            if ($role && $role['is_super_admin']) {
                $error = 'Cannot modify Super Admin role';
            } else {
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Update role details
                    $stmt = $pdo->prepare("UPDATE roles SET name = ?, description = ? WHERE id = ?");
                    $stmt->execute([$_POST['name'], $_POST['description'], $_POST['role_id']]);
                    
                    // Delete existing permissions
                    $stmt = $pdo->prepare("DELETE FROM role_permissions WHERE role_id = ?");
                    $stmt->execute([$_POST['role_id']]);
                    
                    // Insert new permissions
                    if (isset($_POST['permissions']) && is_array($_POST['permissions'])) {
                        $stmt = $pdo->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
                        foreach ($_POST['permissions'] as $permission_id) {
                            $stmt->execute([$_POST['role_id'], $permission_id]);
                        }
                    }
                    
                    $pdo->commit();
                    logSystemAction('role_updated', 'Updated role: ' . $_POST['name']);
                    $success = 'Role updated successfully!';
                } catch (PDOException $e) {
                    $pdo->rollBack();
                    throw $e;
                }
            }
        }
    } catch (PDOException $e) {
        $error = 'Error saving role: ' . $e->getMessage();
    }
}

// Handle role deletion
if (isset($_POST['delete_role'])) {
    try {
        // Check if trying to delete super admin role
        $stmt = $pdo->prepare("SELECT is_super_admin FROM roles WHERE id = ?");
        $stmt->execute([$_POST['delete_role']]);
        $role = $stmt->fetch();
        
        if ($role && $role['is_super_admin']) {
            $error = 'Cannot delete Super Admin role';
        } else {
            // Check if role is in use
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE role_id = ?");
            $stmt->execute([$_POST['delete_role']]);
            if ($stmt->fetchColumn() > 0) {
                $error = 'Cannot delete role: It is assigned to one or more users.';
            } else {
                $stmt = $pdo->prepare("DELETE FROM roles WHERE id = ?");
                $stmt->execute([$_POST['delete_role']]);
                logSystemAction('role_deleted', 'Deleted role ID: ' . $_POST['delete_role']);
                $success = 'Role deleted successfully!';
            }
        }
    } catch (PDOException $e) {
        $error = 'Error deleting role: ' . $e->getMessage();
    }
}

// Get all roles with their permissions
$stmt = $pdo->query("
    SELECT r.*, GROUP_CONCAT(p.name) as permissions
    FROM roles r
    LEFT JOIN role_permissions rp ON r.id = rp.role_id
    LEFT JOIN permissions p ON rp.permission_id = p.id
    GROUP BY r.id
    ORDER BY r.name
");
$roles = $stmt->fetchAll();

// Get all permissions for the form
$stmt = $pdo->query("SELECT * FROM permissions ORDER BY name");
$permissions = $stmt->fetchAll();

// Ensure Super Admin always has all permissions
foreach ($roles as &$role) {
    if (!empty($role['is_super_admin'])) {
        $role['permissions'] = implode(',', array_column($permissions, 'name'));
    }
}
unset($role);

// Add Supplier Manager role if it doesn't exist
$stmt = $pdo->prepare("SELECT id FROM roles WHERE name = 'Supplier Manager'");
$stmt->execute();
if ($stmt->rowCount() === 0) {
    $stmt = $pdo->prepare("INSERT INTO roles (name, description) VALUES ('Supplier Manager', 'Can add and search suppliers')");
    $stmt->execute();
    $role_id = $pdo->lastInsertId();

    // Get permission IDs for add_supplier and search_suppliers
    $stmt = $pdo->prepare("SELECT id FROM permissions WHERE name IN ('add_supplier', 'search_suppliers')");
    $stmt->execute();
    $permission_ids = $stmt->fetchAll(PDO::FETCH_COLUMN);

    // Assign permissions to the role
    $stmt = $pdo->prepare("INSERT INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
    foreach ($permission_ids as $permission_id) {
        $stmt->execute([$role_id, $permission_id]);
    }
}

// Ensure Super Admin has ALL permissions
$stmt = $pdo->prepare("SELECT id FROM roles WHERE is_super_admin = 1");
$stmt->execute();
$superAdmin = $stmt->fetch();
if ($superAdmin) {
    // Get ALL permissions from the permissions table
    $stmt = $pdo->prepare("SELECT id FROM permissions");
    $stmt->execute();
    $all_permission_ids = $stmt->fetchAll(PDO::FETCH_COLUMN);
    
    // Assign ALL permissions to Super Admin (using INSERT IGNORE to avoid duplicates)
    $stmt = $pdo->prepare("INSERT IGNORE INTO role_permissions (role_id, permission_id) VALUES (?, ?)");
    foreach ($all_permission_ids as $permission_id) {
        $stmt->execute([$superAdmin['id'], $permission_id]);
    }
}

// Add 'Allow Credit Sale' permission if not present
$allowCreditSalePerm = null;
foreach ($permissions as $perm) {
    if ($perm['name'] === 'allow_credit_sale') {
        $allowCreditSalePerm = $perm;
        break;
    }
}
if (!$allowCreditSalePerm) {
    // Insert the permission if it doesn't exist
    $stmt = $pdo->prepare("INSERT INTO permissions (name, description) VALUES (?, ?)");
    $stmt->execute(['allow_credit_sale', 'Allow user to process credit sales']);
    // Refresh permissions
    $permissions = $pdo->query("SELECT * FROM permissions ORDER BY name")->fetchAll();
}

// Add missing permissions if not present
$missingPermissions = [
    'process_credit_payment' => 'Process credit payments',
    'waive_credit' => 'Waive credit amounts',
    'send_reminder' => 'Send payment reminders',
    'add_credit_item' => 'Add credit items',
    'manage_categories' => 'Manage product categories',
    'add_product' => 'Add new products',
    'edit_product' => 'Edit existing products',
    'delete_product' => 'Delete products',
    'enable_unpaid_credit_section' => 'Enable Unpaid Credit Section in Sales Receipts'
];

foreach ($missingPermissions as $permName => $permDesc) {
    $permExists = false;
    foreach ($permissions as $perm) {
        if ($perm['name'] === $permName) {
            $permExists = true;
            break;
        }
    }
    if (!$permExists) {
        // Insert the permission if it doesn't exist
        $stmt = $pdo->prepare("INSERT INTO permissions (name, description) VALUES (?, ?)");
        $stmt->execute([$permName, $permDesc]);
    }
}

// Add missing permissions for invoice, delivery, and quotation actions if not present
$actionPermissions = [
    // Invoice
    ['add_invoice', 'Add new invoices'],
    ['view_invoice', 'View invoices'],
    ['edit_invoice', 'Edit invoices'],
    ['pay_invoice', 'Pay invoices'],
    ['print_invoice', 'Print invoices'],
    ['send_invoice_reminder', 'Send invoice reminders'],
    ['delete_invoice', 'Delete invoices'],
    // Delivery
    ['add_delivery', 'Add new deliveries'],
    ['view_delivery', 'View deliveries'],
    ['edit_delivery', 'Edit deliveries'],
    ['mark_delivery_delivered', 'Mark deliveries as delivered'],
    ['print_delivery', 'Print deliveries'],
    ['delete_delivery', 'Delete deliveries'],
    // Quotation
    ['add_quotation', 'Add new quotations'],
    ['view_quotation', 'View quotations'],
    ['edit_quotation', 'Edit quotations'],
    ['convert_quotation_to_invoice', 'Convert quotation to invoice'],
    ['print_quotation', 'Print quotations'],
    ['delete_quotation', 'Delete quotations'],
];
foreach ($actionPermissions as [$permName, $permDesc]) {
    $exists = false;
    foreach ($permissions as $perm) {
        if ($perm['name'] === $permName) {
            $exists = true;
            break;
        }
    }
    if (!$exists) {
        $stmt = $pdo->prepare("INSERT INTO permissions (name, description) VALUES (?, ?)");
        $stmt->execute([$permName, $permDesc]);
    }
}
// Refresh permissions
$permissions = $pdo->query("SELECT * FROM permissions ORDER BY name")->fetchAll();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Roles - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <!-- Session Timeout Management -->
    <script src="../js/session_timeout.js"></script>
    <style>
        body { background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); font-family: 'Inter', sans-serif; }
        .card { border-radius: 16px; box-shadow: 0 2px 12px rgba(37,99,235,0.06); border: none; }
        .card-header { background: none; border-bottom: none; }
        .btn-primary, .btn-warning, .btn-info, .btn-danger { border-radius: 8px; font-weight: 600; }
        .table { background: #fff; border-radius: 12px; overflow: hidden; }
        .table th, .table td { vertical-align: middle; }
        .table-striped > tbody > tr:nth-of-type(odd) { background-color: #f3f4f6; }
        .modal-content { border-radius: 16px; }
        .form-label { font-weight: 600; color: #2563eb; }
        .modal-title { color: #2563eb; font-weight: 700; }
        .summary-title { font-size: 1.1rem; color: #64748b; font-weight: 600; }
        .summary-value { font-size: 1.7rem; font-weight: 700; color: #222; }
        .section-header { margin-bottom: 0.75rem; padding-bottom: 0.5rem; border-bottom: 1px solid #e2e8f0; }
        .section-header .form-check-label { font-size: 0.95rem; color: #1e293b; margin-bottom: 0; }
        .section-title {
            font-size: 1rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.75rem;
            display: flex;
            align-items: center;
        }
        .info-section {
            background: white;
            border-radius: 12px;
            padding: 1rem;
            border: 1px solid #e2e8f0;
            margin-bottom: 1rem;
        }
        .permissions-section {
            background: white;
            border-radius: 12px;
            padding: 1rem;
            border: 1px solid #e2e8f0;
        }
        .quick-actions {
            background: #f1f5f9;
            border-radius: 8px;
            padding: 0.75rem;
            margin-bottom: 0.75rem;
        }
        .permission-section {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 1rem;
            border-left: 3px solid #e9ecef;
            transition: all 0.3s ease;
        }
        
        .permission-section:hover {
            background: #f1f3f4;
            border-left-color: #2563eb;
            box-shadow: 0 1px 4px rgba(37, 99, 235, 0.1);
        }
        
        .permission-list {
            max-height: 120px;
            overflow-y: auto;
            padding-right: 0.25rem;
        }
        
        .permission-item {
            margin-bottom: 0.5rem;
            padding: 0.5rem;
            border-radius: 6px;
            transition: background-color 0.2s ease;
            border: 1px solid #e2e8f0;
            background: white;
        }
        
        .permission-item:hover {
            background-color: rgba(37, 99, 235, 0.05);
            border-color: #2563eb;
        }
        
        .form-check-input {
            margin-right: 0.75rem;
            transform: scale(1.1);
        }
        
        .form-check-input:checked {
            background-color: #2563eb;
            border-color: #2563eb;
        }
        
        .form-check-label {
            font-size: 0.9rem;
            color: #374151;
            font-weight: 500;
            cursor: pointer;
            line-height: 1.4;
            margin-left: 0.25rem;
        }
        
        .form-check-input:checked + .form-check-label {
            color: #1e293b;
            font-weight: 600;
        }
        
        /* Scrollbar Styling */
        .permission-list::-webkit-scrollbar {
            width: 4px;
        }
        
        .permission-list::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 2px;
        }
        
        .permission-list::-webkit-scrollbar-thumb {
            background: #c1c1c1;
            border-radius: 2px;
        }
        
        .permission-list::-webkit-scrollbar-thumb:hover {
            background: #a8a8a8;
        }
        
        /* Responsive adjustments */
        @media (max-width: 768px) {
            .permission-section {
                padding: 0.75rem;
            }
            
            .permission-list {
                max-height: 100px;
            }
            
            .permission-item {
                padding: 0.2rem 0.4rem;
            }
        }

        /* Modern Modal Styles */
        .modern-modal {
            border: none;
            border-radius: 16px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.15);
            overflow: hidden;
        }
        
        .modern-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 1.5rem;
        }
        
        .modal-icon {
            width: 48px;
            height: 48px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
        }
        
        .modern-body {
            padding: 1.5rem;
            background: #f8fafc;
        }
        
        .modern-footer {
            background: white;
            border-top: 1px solid #e2e8f0;
            padding: 1rem 1.5rem;
        }
        
        .modern-btn {
            border-radius: 8px;
            padding: 0.5rem 1rem;
            font-weight: 500;
            transition: all 0.3s ease;
            border: none;
        }
        
        .modern-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }
        
        .modern-input {
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 0.75rem;
            transition: all 0.3s ease;
        }
        
        .modern-input:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-floating > .form-control:focus ~ label,
        .form-floating > .form-control:not(:placeholder-shown) ~ label {
            color: #667eea;
        }
        
        /* Section Styles */
        .section-title {
            font-size: 1rem;
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.75rem;
            display: flex;
            align-items: center;
        }
        
        .info-section {
            background: white;
            border-radius: 12px;
            padding: 1rem;
            border: 1px solid #e2e8f0;
        }
        
        .permissions-section {
            background: white;
            border-radius: 12px;
            padding: 1rem;
            border: 1px solid #e2e8f0;
        }
        
        .quick-actions {
            background: #f1f5f9;
            border-radius: 8px;
            padding: 0.75rem;
            margin-bottom: 0.75rem;
        }
        
        /* Permission Cards */
        .permission-card {
            background: white;
            border: 1px solid #e2e8f0;
            border-radius: 12px;
            overflow: hidden;
            transition: all 0.3s ease;
            margin-bottom: 0.75rem;
        }
        
        .permission-card:hover {
            border-color: #667eea;
            box-shadow: 0 4px 20px rgba(102, 126, 234, 0.1);
        }
        
        .card-header-modern {
            background: #f8fafc;
            padding: 0.75rem 1rem;
            cursor: pointer;
            transition: all 0.3s ease;
            border-bottom: 1px solid #e2e8f0;
        }
        
        .card-header-modern:hover {
            background: #f1f5f9;
        }
        
        .section-icon {
            width: 32px;
            height: 32px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 0.9rem;
        }
        
        .section-toggle {
            color: #64748b;
            transition: transform 0.3s ease;
        }
        
        .permission-card.expanded .section-toggle {
            transform: rotate(180deg);
        }
        
        .card-body-modern {
            padding: 0.75rem 1.5rem;
            max-height: 180px;
            overflow-y: auto;
            transition: all 0.3s ease;
        }
        
        .permission-card.collapsed .card-body-modern {
            max-height: 0;
            padding: 0 1.5rem;
            overflow: hidden;
        }
        
        .permission-item-modern {
            padding: 0.5rem;
            border-radius: 6px;
            margin-bottom: 0.25rem;
            transition: all 0.2s ease;
            border: 1px solid transparent;
        }
        
        .permission-item-modern:hover {
            background: #f8fafc;
            border-color: #e2e8f0;
        }
        
        .permission-item-modern .form-check {
            margin: 0;
        }
        
        .form-check-input {
            width: 1.1rem;
            height: 1.1rem;
            margin-right: 0.5rem;
            border: 2px solid #cbd5e1;
            border-radius: 4px;
            transition: all 0.2s ease;
        }
        
        .form-check-input:checked {
            background-color: #667eea;
            border-color: #667eea;
        }
        
        .form-check-input:focus {
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .form-check-label {
            font-size: 0.85rem;
            color: #374151;
            font-weight: 500;
            cursor: pointer;
            line-height: 1.3;
        }
        
        .form-check-input:checked + .form-check-label {
            color: #1e293b;
            font-weight: 600;
        }
        
        /* Section Checkbox */
        .section-checkbox .form-check-input {
            width: 1rem;
            height: 1rem;
        }
        
        /* Scrollbar Styling */
        .card-body-modern::-webkit-scrollbar {
            width: 6px;
        }
        
        .card-body-modern::-webkit-scrollbar-track {
            background: #f1f5f9;
            border-radius: 3px;
        }
        
        .card-body-modern::-webkit-scrollbar-thumb {
            background: #cbd5e1;
            border-radius: 3px;
        }
        
        .card-body-modern::-webkit-scrollbar-thumb:hover {
            background: #94a3b8;
        }
        
        /* Animations */
        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .permission-card {
            animation: slideDown 0.3s ease;
        }
        
        /* Responsive Design */
        @media (max-width: 768px) {
            .modern-body {
                padding: 1rem;
            }
            
            .modern-footer {
                padding: 1rem;
            }
            
            .card-header-modern {
                padding: 0.75rem 1rem;
            }
            
            .card-body-modern {
                padding: 0.75rem 1rem;
                max-height: 150px;
            }
            
            .quick-actions .d-flex {
                flex-direction: column;
                gap: 0.5rem;
            }
            
            .quick-actions .vr {
                display: none;
            }
        }
        
        /* Loading States */
        .modern-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Success States */
        .permission-item-modern.selected {
            background: #f0f9ff;
            border-color: #0ea5e9;
        }
        
        /* Error States */
        .modern-input.is-invalid {
            border-color: #ef4444;
        }
        
        .modern-input.is-invalid:focus {
            box-shadow: 0 0 0 3px rgba(239, 68, 68, 0.1);
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Role Management</h4>
                        <?php if (hasPermission('manage_roles')): ?>
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#roleModal">
                            <i class="fas fa-plus me-2"></i>Add New Role
                        </button>
                        <?php endif; ?>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success"><?php echo $success; ?></div>
                        <?php endif; ?>
                        <?php if ($error): ?>
                            <div class="alert alert-danger"><?php echo $error; ?></div>
                        <?php endif; ?>

                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Role Name</th>
                                        <th>Description</th>
                                        <th>Permissions</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($roles as $role): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($role['name']); ?></td>
                                            <td><?php echo htmlspecialchars($role['description']); ?></td>
                                            <td>
                                                <?php 
                                                if ($role['permissions']) {
                                                    $perms = explode(',', $role['permissions']);
                                                    foreach ($perms as $perm) {
                                                        echo '<span class="badge bg-info me-1">' . htmlspecialchars($perm) . '</span>';
                                                    }
                                                }
                                                ?>
                                            </td>
                                            <td>
                                                <?php if (!$role['is_super_admin'] && hasPermission('manage_roles')): ?>
                                                    <button type="button" class="btn btn-sm btn-primary edit-role" 
                                                            data-role='<?php echo htmlspecialchars(json_encode($role)); ?>'
                                                            data-bs-toggle="modal" data-bs-target="#roleModal"
                                                            data-permission="edit_role" data-permission-label="Edit Role">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <form method="post" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this role?');">
                                                        <input type="hidden" name="delete_role" value="<?php echo $role['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-danger" data-permission="delete_role" data-permission-label="Delete Role">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                    </form>
                                                <?php elseif ($role['is_super_admin']): ?>
                                                    <span class="badge bg-warning">Protected</span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Role Modal -->
    <div class="modal fade" id="roleModal" tabindex="-1" data-bs-backdrop="static">
        <div class="modal-dialog modal-lg modal-dialog-centered">
            <div class="modal-content modern-modal">
                <div class="modal-header modern-header">
                    <div class="d-flex align-items-center">
                        <div class="modal-icon me-3">
                            <i class="fas fa-user-shield"></i>
                </div>
                        <div>
                            <h5 class="modal-title mb-0">Add/Edit Role</h5>
                            <small class="text-muted">Configure role permissions and access levels</small>
                        </div>
                    </div>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="post" class="modern-form">
                    <div class="modal-body modern-body">
                        <input type="hidden" name="role_id" id="role_id">
                        <input type="hidden" name="action" id="action" value="create">
                        
                        <!-- Basic Role Information -->
                        <div class="info-section mb-4">
                            <div class="section-title">
                                <i class="fas fa-info-circle me-2"></i>
                                Basic Information
                        </div>
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <div class="form-floating">
                                        <input type="text" class="form-control modern-input" name="name" id="role_name" placeholder="Role Name" required>
                                        <label for="role_name">Role Name</label>
                        </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-floating">
                                        <textarea class="form-control modern-input" name="description" id="role_description" placeholder="Description" rows="2"></textarea>
                                        <label for="role_description">Description</label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Permissions Section -->
                        <div class="permissions-section">
                            <div class="section-title">
                                <i class="fas fa-shield-alt me-2"></i>
                                Permissions
                            </div>
                            
                            <!-- Quick Actions -->
                            <div class="quick-actions mb-3">
                                <div class="d-flex gap-2 flex-wrap">
                                    <button type="button" class="btn btn-primary btn-sm modern-btn" onclick="selectAllPermissions()">
                                        <i class="fas fa-check-double me-1"></i>Select All
                                    </button>
                                    <button type="button" class="btn btn-outline-secondary btn-sm modern-btn" onclick="deselectAllPermissions()">
                                        <i class="fas fa-times me-1"></i>Clear All
                                    </button>
                                    <div class="vr"></div>
                                    <button type="button" class="btn btn-outline-info btn-sm modern-btn" onclick="expandAllSections()">
                                        <i class="fas fa-expand me-1"></i>Expand All
                                    </button>
                                    <button type="button" class="btn btn-outline-info btn-sm modern-btn" onclick="collapseAllSections()">
                                        <i class="fas fa-compress me-1"></i>Collapse All
                                    </button>
                                </div>
                            </div>

                            <div class="row g-3">


                               <!-- Transactions Permissions -->
   <div class="permission-card" data-section="transactions">
       <div class="card-header-modern" onclick="toggleSection('transactions')">
           <div class="d-flex align-items-center justify-content-between">
               <div class="d-flex align-items-center">
                   <div class="section-checkbox me-2">
                       <input class="form-check-input section-selector" type="checkbox" 
                              id="select_transactions" onchange="toggleSection('transactions')">
                   </div>
                   <div class="section-icon me-2">
                       <i class="fas fa-exchange-alt"></i>
                   </div>
                   <div>
                       <h6 class="section-title mb-0">Transactions</h6>
                       <small class="text-muted">Invoices, Deliveries, Quotations</small>
                   </div>
               </div>
               <div class="section-toggle">
                   <i class="fas fa-chevron-down"></i>
               </div>
           </div>
       </div>
       <div class="card-body-modern" id="transactions_permissions">
           <?php foreach ($permissions as $permission): ?>
               <?php if (in_array($permission['name'], [
                   'view_invoice','edit_invoice','pay_invoice','print_invoice','send_invoice_reminder','delete_invoice',
                   'view_delivery','edit_delivery','mark_delivery_delivered','print_delivery','delete_delivery',
                   'view_quotation','edit_quotation','convert_quotation_to_invoice','print_quotation','delete_quotation'
               ])): ?>
                   <div class="permission-item-modern">
                       <div class="form-check">
                           <input class="form-check-input transactions-perm" type="checkbox" 
                                  name="permissions[]" value="<?php echo $permission['id']; ?>"
                                  id="perm_<?php echo $permission['id']; ?>">
                           <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                               <?php echo htmlspecialchars($permission['description']); ?>
                           </label>
                       </div>
                   </div>
               <?php endif; ?>
           <?php endforeach; ?>
       </div>
   </div>
                                <!-- Left Column -->
                                <div class="col-md-6">
                                    <!-- Sales & POS Permissions -->
                                    <div class="permission-card" data-section="sales">
                                        <div class="card-header-modern" onclick="toggleSection('sales')">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div class="d-flex align-items-center">
                                                    <div class="section-checkbox me-2">
                                                        <input class="form-check-input section-selector" type="checkbox" 
                                                               id="select_sales" onchange="toggleSection('sales')">
                                                    </div>
                                                    <div class="section-icon me-2">
                                                        <i class="fas fa-cash-register"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="section-title mb-0">Sales & POS</h6>
                                                        <small class="text-muted">Point of sale operations</small>
                                                    </div>
                                                </div>
                                                <div class="section-toggle">
                                                    <i class="fas fa-chevron-down"></i>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body-modern" id="sales_permissions">
                                    <?php foreach ($permissions as $permission): ?>
                                                <?php if (in_array($permission['name'], ['view_sales', 'create_sales', 'edit_sales', 'delete_sales', 'process_refunds', 'allow_credit_sale'])): ?>
                                                    <div class="permission-item-modern">
                                                        <div class="form-check">
                                                            <input class="form-check-input sales-perm" type="checkbox" 
                                                       name="permissions[]" value="<?php echo $permission['id']; ?>"
                                                       id="perm_<?php echo $permission['id']; ?>">
                                                <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                                                    <?php echo htmlspecialchars($permission['description']); ?>
                                                </label>
                                                        </div>
                                            </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                                    </div>

                                    <!-- Inventory & Products Permissions -->
                                    <div class="permission-card" data-section="inventory">
                                        <div class="card-header-modern" onclick="toggleSection('inventory')">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div class="d-flex align-items-center">
                                                    <div class="section-checkbox me-2">
                                                        <input class="form-check-input section-selector" type="checkbox" 
                                                               id="select_inventory" onchange="toggleSection('inventory')">
                                                    </div>
                                                    <div class="section-icon me-2">
                                                        <i class="fas fa-box"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="section-title mb-0">Inventory & Products</h6>
                                                        <small class="text-muted">Stock and product management</small>
                                                    </div>
                                                </div>
                                                <div class="section-toggle">
                                                    <i class="fas fa-chevron-down"></i>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body-modern" id="inventory_permissions">
                                    <?php foreach ($permissions as $permission): ?>
                                                <?php if (in_array($permission['name'], ['view_inventory', 'manage_inventory', 'adjust_stock', 'view_products', 'manage_products', 'view_stock', 'manage_stock', 'add_stock', 'edit_stock', 'delete_stock', 'manage_categories', 'add_product', 'edit_product', 'delete_product'])): ?>
                                                    <div class="permission-item-modern">
                                                        <div class="form-check">
                                                            <input class="form-check-input inventory-perm" type="checkbox" 
                                                       name="permissions[]" value="<?php echo $permission['id']; ?>"
                                                       id="perm_<?php echo $permission['id']; ?>">
                                                <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                                                    <?php echo htmlspecialchars($permission['description']); ?>
                                                </label>
                                                        </div>
                                            </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                                    </div>

                                    <!-- Reports Permissions -->
                                    <div class="permission-card" data-section="reports">
                                        <div class="card-header-modern" onclick="toggleSection('reports')">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div class="d-flex align-items-center">
                                                    <div class="section-checkbox me-2">
                                                        <input class="form-check-input section-selector" type="checkbox" 
                                                               id="select_reports" onchange="toggleSection('reports')">
                                                    </div>
                                                    <div class="section-icon me-2">
                                                        <i class="fas fa-chart-line"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="section-title mb-0">Reports & Analytics</h6>
                                                        <small class="text-muted">Data analysis and reporting</small>
                                                    </div>
                                                </div>
                                                <div class="section-toggle">
                                                    <i class="fas fa-chevron-down"></i>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body-modern" id="reports_permissions">
                                            <?php foreach ($permissions as $permission): ?>
                                                <?php if (in_array($permission['name'], ['view_reports', 'export_reports', 'view_dashboard'])): ?>
                                                    <div class="permission-item-modern">
                                                        <div class="form-check">
                                                            <input class="form-check-input reports-perm" type="checkbox" 
                                                                   name="permissions[]" value="<?php echo $permission['id']; ?>"
                                                                   id="perm_<?php echo $permission['id']; ?>">
                                                            <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                                                                <?php echo htmlspecialchars($permission['description']); ?>
                                                            </label>
                                                        </div>
                                                    </div>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                </div>

                                <!-- Right Column -->
                                <div class="col-md-6">
                                    <!-- Customer Management Permissions -->
                                    <div class="permission-card" data-section="customers">
                                        <div class="card-header-modern" onclick="toggleSection('customers')">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div class="d-flex align-items-center">
                                                    <div class="section-checkbox me-2">
                                                        <input class="form-check-input section-selector" type="checkbox" 
                                                               id="select_customers" onchange="toggleSection('customers')">
                                                    </div>
                                                    <div class="section-icon me-2">
                                                        <i class="fas fa-users"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="section-title mb-0">Customer Management</h6>
                                                        <small class="text-muted">Customer data and credit management</small>
                                                    </div>
                                                </div>
                                                <div class="section-toggle">
                                                    <i class="fas fa-chevron-down"></i>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body-modern" id="customers_permissions">
                                    <?php foreach ($permissions as $permission): ?>
                                                <?php if (in_array($permission['name'], ['view_customers', 'manage_customers', 'add_customer', 'edit_customer', 'delete_customer', 'manage_credit', 'process_credit_payment', 'waive_credit', 'send_reminder', 'add_credit_item'])): ?>
                                                    <div class="permission-item-modern">
                                                        <div class="form-check">
                                                            <input class="form-check-input customers-perm" type="checkbox" 
                                                           name="permissions[]" value="<?php echo $permission['id']; ?>"
                                                           id="perm_<?php echo $permission['id']; ?>">
                                                    <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                                                        <?php echo htmlspecialchars($permission['description']); ?>
                                                    </label>
                                                        </div>
                                                </div>
                                            <?php endif; ?>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>

                                    <!-- Supplier Management Permissions -->
                                    <div class="permission-card" data-section="suppliers">
                                        <div class="card-header-modern" onclick="toggleSection('suppliers')">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div class="d-flex align-items-center">
                                                    <div class="section-checkbox me-2">
                                                        <input class="form-check-input section-selector" type="checkbox" 
                                                               id="select_suppliers" onchange="toggleSection('suppliers')">
                                                    </div>
                                                    <div class="section-icon me-2">
                                                        <i class="fas fa-truck"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="section-title mb-0">Supplier Management</h6>
                                                        <small class="text-muted">Supplier and vendor management</small>
                                                    </div>
                                                </div>
                                                <div class="section-toggle">
                                                    <i class="fas fa-chevron-down"></i>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body-modern" id="suppliers_permissions">
                                            <?php foreach ($permissions as $permission): ?>
                                                <?php if (in_array($permission['name'], ['view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier', 'search_suppliers', 'process_supplier_payment', 'add_supplier_item'])): ?>
                                                    <div class="permission-item-modern">
                                                        <div class="form-check">
                                                            <input class="form-check-input suppliers-perm" type="checkbox" 
                                                                   name="permissions[]" value="<?php echo $permission['id']; ?>"
                                                                   id="perm_<?php echo $permission['id']; ?>">
                                                            <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                                                                <?php echo htmlspecialchars($permission['description']); ?>
                                                            </label>
                                                        </div>
                                                    </div>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </div>
                                    </div>

                                    <!-- System Administration Permissions -->
                                    <div class="permission-card" data-section="admin">
                                        <div class="card-header-modern" onclick="toggleSection('admin')">
                                            <div class="d-flex align-items-center justify-content-between">
                                                <div class="d-flex align-items-center">
                                                    <div class="section-checkbox me-2">
                                                        <input class="form-check-input section-selector" type="checkbox" 
                                                               id="select_admin" onchange="toggleSection('admin')">
                                                    </div>
                                                    <div class="section-icon me-2">
                                                        <i class="fas fa-cog"></i>
                                                    </div>
                                                    <div>
                                                        <h6 class="section-title mb-0">System Administration</h6>
                                                        <small class="text-muted">System configuration and management</small>
                                                    </div>
                                                </div>
                                                <div class="section-toggle">
                                                    <i class="fas fa-chevron-down"></i>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="card-body-modern" id="admin_permissions">
                                            <?php foreach ($permissions as $permission): ?>
                                                <?php if (in_array($permission['name'], ['manage_users', 'add_user', 'edit_user', 'delete_user', 'manage_roles', 'edit_role', 'delete_role', 'view_settings', 'manage_settings', 'view_logs', 'manage_logs', 'view_system_logs', 'manage_email_settings', 'manage_mpesa_settings', 'manage_sms_settings', 'create_qr_code', 'enable_unpaid_credit_section'])): ?>
                                                    <div class="permission-item-modern">
                                    <div class="form-check">
                                                            <input class="form-check-input admin-perm" type="checkbox" 
                                                                   name="permissions[]" value="<?php echo $permission['id']; ?>"
                                                                   id="perm_<?php echo $permission['id']; ?>">
                                                            <label class="form-check-label" for="perm_<?php echo $permission['id']; ?>">
                                                                <?php echo htmlspecialchars($permission['description']); ?>
                                        </label>
                                    </div>
                                </div>
                                                <?php endif; ?>
                                            <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer modern-footer">
                        <button type="button" class="btn btn-light modern-btn" data-bs-dismiss="modal">
                            <i class="fas fa-times me-1"></i>Cancel
                        </button>
                        <button type="submit" class="btn btn-primary modern-btn" <?php echo !hasPermission('manage_roles') ? 'disabled' : ''; ?>>
                            <i class="fas fa-save me-1"></i>Save Role
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Add/Edit Role Modal -->
    <div class="modal fade" id="addEditRoleModal" tabindex="-1">
      <div class="modal-dialog">
        <div class="modal-content">
          <div class="modal-header">
            <h5 class="modal-title" id="roleModalTitle">Add/Edit Role</h5>
            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
          </div>
          <form id="roleForm">
            <div class="modal-body">
              <div class="mb-3">
                <label for="roleName" class="form-label">Role Name</label>
                <input type="text" class="form-control" id="roleName" name="roleName" required>
              </div>
              <div class="mb-3">
                <label for="roleDescription" class="form-label">Description</label>
                <textarea class="form-control" id="roleDescription" name="roleDescription" rows="2"></textarea>
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
              <button type="submit" class="btn btn-primary">Save Role</button>
            </div>
          </form>
        </div>
      </div>
    </div>

    <!-- Access Restricted Modal for Role Actions -->
    <div class="modal fade" id="accessRestrictedModal" tabindex="-1" aria-labelledby="accessRestrictedModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content" style="border-radius: 20px; border: none; box-shadow: 0 20px 60px rgba(0,0,0,0.15);">
                <div class="modal-header" style="background: linear-gradient(135deg, #dc2626 0%, #b91c1c 100%); color: white; border-radius: 20px 20px 0 0; border: none; padding: 1.5rem;">
                    <div class="d-flex align-items-center">
                        <div class="me-3" style="width: 50px; height: 50px; background: rgba(255,255,255,0.2); border-radius: 50%; display: flex; align-items: center; justify-content: center;">
                            <i class="fas fa-lock" style="font-size: 1.5rem;"></i>
                        </div>
                        <div>
                            <h5 class="modal-title mb-0" id="accessRestrictedModalLabel" style="font-weight: 700; font-size: 1.3rem;">Access Restricted</h5>
                            <small style="opacity: 0.9;">Role Change Confirmation</small>
                        </div>
                    </div>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" style="padding: 2rem; text-align: center;">
                    <div class="mb-4">
                        <div style="width: 80px; height: 80px; background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%); border: 3px solid #dc2626; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin: 0 auto 1rem;">
                            <i class="fas fa-exclamation-triangle" style="font-size: 2rem; color: #dc2626;"></i>
                        </div>
                        <h6 style="font-weight: 700; color: #1f2937; margin-bottom: 1rem;">Are you sure you want to save changes to this role?</h6>
                        <p style="color: #6b7280; font-size: 1rem; line-height: 1.5;">
                            Modifying roles can affect user permissions across the system. Please confirm you want to proceed.
                        </p>
                    </div>
                </div>
                <div class="modal-footer" style="border: none; padding: 1.5rem 2rem 2rem 2rem; background: #f9fafb;">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" 
                            style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600;">Cancel</button>
                    <button type="button" class="btn btn-primary" id="confirmRoleSave"
                            style="border-radius: 10px; padding: 0.75rem 1.5rem; font-weight: 600; background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%); border: none;">
                        <i class="fas fa-check me-2"></i>Yes, Save
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Initialize modal functionality
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize all permission cards as expanded
            document.querySelectorAll('.permission-card').forEach(card => {
                card.classList.add('expanded');
            });
            
            // Add click handlers for section toggles
            document.querySelectorAll('.card-header-modern').forEach(header => {
                header.addEventListener('click', function(e) {
                    // Don't toggle if clicking on checkbox
                    if (e.target.type === 'checkbox') return;
                    
                    const card = this.closest('.permission-card');
                    const section = card.dataset.section;
                    toggleSectionVisibility(section);
                });
            });
        });

        // Handle edit role
        document.querySelectorAll('.edit-role').forEach(button => {
            button.addEventListener('click', function() {
                const role = JSON.parse(this.dataset.role);
                const form = document.querySelector('#roleModal form');
                form.querySelector('[name="action"]').value = 'update';
                form.querySelector('[name="role_id"]').value = role.id;
                form.querySelector('[name="name"]').value = role.name;
                form.querySelector('[name="description"]').value = role.description;
                
                // Reset all checkboxes
                form.querySelectorAll('[name="permissions[]"]').forEach(checkbox => {
                    checkbox.checked = false;
                });
                
                // Reset section selectors
                document.querySelectorAll('.section-selector').forEach(checkbox => {
                    checkbox.checked = false;
                    checkbox.indeterminate = false;
                });
                
                // Get role permissions
                fetch(`get_role_permissions.php?role_id=${role.id}`)
                    .then(response => response.json())
                    .then(permissions => {
                        permissions.forEach(permissionId => {
                            const checkbox = form.querySelector(`[name="permissions[]"][value="${permissionId}"]`);
                            if (checkbox) checkbox.checked = true;
                        });
                        
                        // Update section selectors based on selected permissions
                        updateSectionSelectors();
                    })
                    .catch(error => console.error('Error fetching permissions:', error));
            });
        });

        // Reset form when modal is closed
        document.getElementById('roleModal').addEventListener('hidden.bs.modal', function() {
            const form = this.querySelector('form');
            form.reset();
            form.querySelector('[name="action"]').value = 'create';
            form.querySelector('[name="role_id"]').value = '';
            
            // Reset section selectors
            document.querySelectorAll('.section-selector').forEach(checkbox => {
                checkbox.checked = false;
                checkbox.indeterminate = false;
            });
            
            // Expand all sections
            document.querySelectorAll('.permission-card').forEach(card => {
                card.classList.add('expanded');
                card.classList.remove('collapsed');
            });
        });

        // Select all permissions
        function selectAllPermissions() {
            document.querySelectorAll('[name="permissions[]"]').forEach(checkbox => {
                checkbox.checked = true;
            });
            updateSectionSelectors();
            showNotification('All permissions selected', 'success');
        }

        // Deselect all permissions
        function deselectAllPermissions() {
            document.querySelectorAll('[name="permissions[]"]').forEach(checkbox => {
                checkbox.checked = false;
            });
            updateSectionSelectors();
            showNotification('All permissions cleared', 'info');
        }

        // Toggle section permissions
        function toggleSection(section) {
            const sectionCheckbox = document.getElementById(`select_${section}`);
            const permissionCheckboxes = document.querySelectorAll(`.${section}-perm`);
            
            permissionCheckboxes.forEach(checkbox => {
                checkbox.checked = sectionCheckbox.checked;
            });
            
            updateSectionSelectors();
            
            const action = sectionCheckbox.checked ? 'selected' : 'cleared';
            showNotification(`${section.charAt(0).toUpperCase() + section.slice(1)} permissions ${action}`, 'success');
        }

        // Toggle section visibility (expand/collapse)
        function toggleSectionVisibility(section) {
            const card = document.querySelector(`[data-section="${section}"]`);
            const isExpanded = card.classList.contains('expanded');
            
            if (isExpanded) {
                card.classList.remove('expanded');
                card.classList.add('collapsed');
            } else {
                card.classList.remove('collapsed');
                card.classList.add('expanded');
            }
        }

        // Expand all sections
        function expandAllSections() {
            document.querySelectorAll('.permission-card').forEach(card => {
                card.classList.add('expanded');
                card.classList.remove('collapsed');
            });
            showNotification('All sections expanded', 'info');
        }

        // Collapse all sections
        function collapseAllSections() {
            document.querySelectorAll('.permission-card').forEach(card => {
                card.classList.remove('expanded');
                card.classList.add('collapsed');
            });
            showNotification('All sections collapsed', 'info');
        }

        // Update section selectors based on individual permissions
        function updateSectionSelectors() {
            const sections = ['sales', 'inventory', 'reports', 'customers', 'suppliers', 'admin'];
            
            sections.forEach(section => {
                const sectionCheckbox = document.getElementById(`select_${section}`);
                const permissionCheckboxes = document.querySelectorAll(`.${section}-perm`);
                const checkedPermissions = document.querySelectorAll(`.${section}-perm:checked`);
                
                if (checkedPermissions.length === 0) {
                    sectionCheckbox.checked = false;
                    sectionCheckbox.indeterminate = false;
                } else if (checkedPermissions.length === permissionCheckboxes.length) {
                    sectionCheckbox.checked = true;
                    sectionCheckbox.indeterminate = false;
                } else {
                    sectionCheckbox.checked = false;
                    sectionCheckbox.indeterminate = true;
                }
            });
        }

        // Show notification
        function showNotification(message, type = 'info') {
            // Create notification element
            const notification = document.createElement('div');
            notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
            notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
            notification.innerHTML = `
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            // Add to body
            document.body.appendChild(notification);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.remove();
                }
            }, 3000);
        }

        // Listen for changes in individual permissions to update section selectors
        document.addEventListener('change', function(e) {
            if (e.target.name === 'permissions[]') {
                updateSectionSelectors();
                
                // Add visual feedback
                const item = e.target.closest('.permission-item-modern');
                if (e.target.checked) {
                    item.classList.add('selected');
                    setTimeout(() => item.classList.remove('selected'), 1000);
                }
            }
        });

        // Form validation
        document.querySelector('#roleModal form').addEventListener('submit', function(e) {
            const roleName = this.querySelector('[name="name"]').value.trim();
            const selectedPermissions = this.querySelectorAll('[name="permissions[]"]:checked');
            
            if (!roleName) {
                e.preventDefault();
                showNotification('Please enter a role name', 'warning');
                this.querySelector('[name="name"]').classList.add('is-invalid');
                return;
            }
            
            if (selectedPermissions.length === 0) {
                e.preventDefault();
                showNotification('Please select at least one permission', 'warning');
                return;
            }
            
            // Show loading state
            const submitBtn = this.querySelector('button[type="submit"]');
            const originalText = submitBtn.innerHTML;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Saving...';
            submitBtn.disabled = true;
            
            // Re-enable after a delay (in case of validation errors)
            setTimeout(() => {
                submitBtn.innerHTML = originalText;
                submitBtn.disabled = false;
            }, 3000);
        });

        // Intercept role form submission to show Access Restricted modal
        let pendingRoleFormSubmit = false;
        document.getElementById('roleForm').addEventListener('submit', function(e) {
          if (!pendingRoleFormSubmit) {
            e.preventDefault();
            const accessModal = new bootstrap.Modal(document.getElementById('accessRestrictedModal'));
            accessModal.show();
          }
        });
        document.getElementById('confirmRoleSave').addEventListener('click', function() {
          pendingRoleFormSubmit = true;
          document.getElementById('accessRestrictedModal').classList.remove('show');
          document.getElementById('accessRestrictedModal').style.display = 'none';
          document.body.classList.remove('modal-open');
          document.getElementById('roleForm').submit();
          setTimeout(() => { pendingRoleFormSubmit = false; }, 1000);
        });
    </script>
    
    <?php 
    // Add permission modal components
    echo getPermissionModalCSS();
    echo getPermissionModalHTML();
    echo getPermissionModalJS();
    ?>
    
    <?php echo getUniversalPermissionModalHTML(); ?>
    <?php echo getUniversalPermissionModalJS(); ?>
    
    <?php include '../includes/footer.php'; ?>
</body>
</html> 